<?php
/******************************************************************************
 *
 * Subrion Real Estate Classifieds Software
 * Copyright (C) 2018 Intelliants, LLC <https://intelliants.com>
 *
 * This file is part of Subrion Real Estate Classifieds Software.
 *
 * This program is a commercial software and any kind of using it must agree
 * to the license, see <https://subrion.pro/license.html>.
 *
 * This copyright notice may not be removed from the software source without
 * the permission of Subrion respective owners.
 *
 *
 * @link https://subrion.pro/product/real-estate.html
 *
 ******************************************************************************/

class iaLocation extends abstractModuleFront implements iaRealEstateModule
{
    const VALUES_DELIMITER_CHAR = ',';
    const ID_ROOT = 1;

    const COLUMN_TITLE = 'title';
    const COLUMN_URL = 'url';
    const COLUMN_CODE = 'code';
    const FAKE_COLUMN_ABBREVIATION = 'abbreviation';

    const NS_NODE_ID = 'nid';
    const NS_PARENT = 'parent';
    const NS_LEFT = 'left';
    const NS_RIGHT = 'right';
    const NS_LEVEL = 'level';

    const COUNTER_DECREMENT = '-';
    const COUNTER_INCREMENT = '+';

    const SQL_CONDITION_BETWEEN = 'BETWEEN';
    const SQL_CONDITION_EQUAL = '=';
    const SQL_CONDITION_NOT_EQUAL = '!=';
    const SQL_CONDITION_LIKE = 'LIKE';

    private $_sqlSelectPattern = 'SELECT :columns FROM `:data_table` d LEFT JOIN `:tree_table` t ON (t.`nid` = d.`id`) :extra_joins WHERE :conditions ORDER BY :order';

    protected static $_table = 'locations';
    protected static $_tableTree = 'locations_tree';

    protected $_itemName = 'location';

    protected $_statuses = [iaCore::STATUS_ACTIVE, iaCore::STATUS_INACTIVE];


    public static function getTableTree($withPrefix = false)
    {
        return ($withPrefix ? iaCore::instance()->iaDb->prefix : '') . self::$_tableTree;
    }

    public function getUrl(array $data)
    {
        return $this->getInfo('url') . $data[self::COLUMN_URL];
    }

    // util functions
    protected function _getNode($nodeId)
    {
        return $this->iaDb->row(iaDb::ALL_COLUMNS_SELECTION, iaDb::convertIds($nodeId, self::NS_NODE_ID),
            self::getTableTree());
    }

    public function search(
        array $statements,
        $columns = null,
        $order = ['field' => self::COLUMN_TITLE, 'direction' => 'ASC'],
        $start = null,
        $limit = null,
        $condition = 'AND'
    ) {
        $nsColumns = [self::NS_LEFT, self::NS_RIGHT, self::NS_PARENT, self::NS_LEVEL];

        $multilingualFields = $this->iaCore->factory('field')->getMultilingualFields($this->getItemName());

        if (empty($columns)) {
            $columns = [iaDb::ALL_COLUMNS_SELECTION];
        }
        $sqlFields = '';
        foreach ($columns as $column => $alias) {
            if ($alias == iaDb::ALL_COLUMNS_SELECTION) {
                $sqlFields .= 'd.*';
            } else {
                $params = [
                    'table' => in_array($alias, $nsColumns) ? 't' : 'd',
                    'column' => is_numeric($column) ? $alias : $column,
                    'alias' => $alias
                ];

                if (self::FAKE_COLUMN_ABBREVIATION == $alias) {
                    $params['table'] = 'p';
                    $params['column'] = self::COLUMN_CODE;
                } elseif (in_array($alias, $multilingualFields)) {
                    $params['column'].= '_' . $this->iaCore->language['iso'];
                }

                $sqlFields .= iaDb::printf(':table.`:column` `:alias`', $params);
            }
            $sqlFields .= ', ';
        }
        $sqlFields = substr($sqlFields, 0, -2);

        $conditions = [];
        foreach ($statements as $options) {
            $options['table'] = in_array($options['column'], $nsColumns) ? 't' : 'd';
            if ($options['column'] == self::FAKE_COLUMN_ABBREVIATION) {
                $options['column'] = self::COLUMN_CODE;
                $options['table'] = 'p';
            }

            if (in_array($options['column'], $multilingualFields)) {
                $options['column'] .= '_' . $this->iaCore->language['iso'];
            }
            switch ($options['cond']) {
                case self::SQL_CONDITION_BETWEEN:
                    $pattern = ':table.`:column` BETWEEN :lower AND :upper';
                    $options = array_merge($options, [
                        'lower' => $options['value'][0],
                        'upper' => $options['value'][1]
                    ]);
                    break;
                case self::SQL_CONDITION_LIKE:
                    $pattern = ":table.`:column` LIKE ':value%'";
                    break;
                default:
                    $pattern = ":table.`:column` :cond ':value'";
            }
            $conditions[] = iaDb::printf($pattern, $options);
        }

        $sql = $this->_sqlSelectPattern;
        $sql = iaDb::printf($sql, [
            'extra_joins' => in_array(self::FAKE_COLUMN_ABBREVIATION, $columns)
                ? "LEFT JOIN `:data_table` p ON (t.parent = p.id AND p.status = '" . iaCore::STATUS_ACTIVE . "')"
                : ''
        ]);

        if (is_array($order)) {
            $order['table'] = in_array($order['field'], $nsColumns) ? 't' : 'd';
            if (in_array($order['field'], $multilingualFields)) {
                $order['field'].= '_' . $this->iaCore->language['iso'];
            }
            $order = iaDb::printf(':table.`:field` :direction', $order);
        }

        $params = [
            'data_table' => self::getTable(true),
            'tree_table' => self::getTableTree(true),
            'columns' => $sqlFields,
            'conditions' => $conditions ? implode(' ' . $condition . ' ', $conditions) : iaDb::EMPTY_CONDITION,
            'order' => $order
        ];

        if ($start !== null && $limit !== null) {
            $sql .= ' LIMIT :start, :limit';
            $params['start'] = $start;
            $params['limit'] = $limit;
        }

        $sql = iaDb::printf($sql, $params);

        $rows = $this->iaDb->getAll($sql);

        $this->_processValues($rows);

        return $rows;
    }

    // designed to return the only row
    protected function _get($conditions, array $columns = [iaDb::ALL_COLUMNS_SELECTION])
    {
        $sqlFields = (string)'';
        foreach ($columns as $column => $alias) {
            $sqlFields .= is_numeric($column) ? ($alias == iaDb::ALL_COLUMNS_SELECTION ? 'd.' . iaDb::ALL_COLUMNS_SELECTION : '`' . $alias . '`') : iaDb::printf('`:column` `:alias`',
                ['column' => $column, 'alias' => $alias]);
            $sqlFields .= ', ';
        }
        $sqlFields = substr($sqlFields, 0, -2);

        $sqlWhere = (string)iaDb::EMPTY_CONDITION . ' ';
        foreach ($conditions as $column => $value) {
            $sqlWhere .= ' AND ';
            $sqlWhere .= in_array($column,
                [self::NS_LEFT, self::NS_RIGHT, self::NS_PARENT, self::NS_LEVEL]) ? 't' : 'd';
            $sqlWhere .= iaDb::printf(".`:column` = ':value'",
                ['column' => $column, 'value' => iaSanitize::sql($value)]);
        }

        $sql = iaDb::printf($this->_sqlSelectPattern, [
            'data_table' => self::getTable(true),
            'tree_table' => self::getTableTree(true),
            'extra_joins' => '',
            'columns' => $sqlFields,
            'conditions' => $sqlWhere,
            'order' => 'd.id'
        ]);

        $row = $this->iaDb->getRow($sql);

        $this->_processValues($row, true);

        return $row;
    }

    // TODO: should this really be separated?
    public function getParentsFlat($parentId)
    {
        if (empty($parentId)) {
            return '';
        }

        $parentNode = $this->_getNode($parentId);

        $sql = <<<SQL
SELECT GROUP_CONCAT(`nid` ORDER BY `level` ASC) `path` FROM `:tree` 
WHERE `left` <= :left AND `right` >= :right AND `level` > 0
SQL;
        $sql = iaDb::printf($sql, [
            'tree' => self::getTableTree(true),
            'left' => $parentNode[self::NS_LEFT],
            'right' => $parentNode[self::NS_RIGHT]
        ]);

        return $this->iaDb->getOne($sql);
    }

    // common system methods
    public function getById($id, $process = true)
    {
        return $this->_get([self::COLUMN_ID => $id], [iaDb::ALL_COLUMNS_SELECTION, self::NS_PARENT]);
    }

    public function getChildrenList(array $params)
    {
        $parentId = empty($params['pid']) ? self::ID_ROOT : $params['pid'];

        if (!$this->_get([self::COLUMN_ID => $parentId, self::COLUMN_STATUS => iaCore::STATUS_ACTIVE],
            [iaDb::ID_COLUMN_SELECTION, self::COLUMN_URL])
        ) {
            return [];
        }

        $parentNode = $this->_getNode($parentId);

        $conditions = [
            ['column' => self::COLUMN_STATUS, 'cond' => self::SQL_CONDITION_EQUAL, 'value' => iaCore::STATUS_ACTIVE],
            [
                'column' => self::NS_LEVEL,
                'cond' => self::SQL_CONDITION_EQUAL,
                'value' => $parentNode[self::NS_LEVEL] + 1
            ]
        ];

        if ($parentNode) {
            $conditions[] = [
                'column' => self::NS_LEFT,
                'cond' => self::SQL_CONDITION_BETWEEN,
                'value' => [$parentNode[self::NS_LEFT], $parentNode[self::NS_RIGHT]]
            ];
        }

        if (isset($params['id']) && is_numeric($params['id']) && $params['id']) {
            $conditions[] = [
                'column' => self::COLUMN_ID,
                'cond' => self::SQL_CONDITION_NOT_EQUAL,
                'value' => $params['id']
            ];
        }

        return $this->search($conditions, [self::COLUMN_ID, self::COLUMN_TITLE, self::COLUMN_URL, 'locked']);
    }

    public function getParents($id)
    {
        $parentNode = $this->_getNode($id);

        $conditions = [
            ['column' => self::NS_LEFT, 'cond' => '<=', 'value' => $parentNode[self::NS_LEFT]],
            ['column' => self::NS_RIGHT, 'cond' => '>=', 'value' => $parentNode[self::NS_RIGHT]],
            ['column' => self::NS_LEVEL, 'cond' => '>', 'value' => 0],
            ['column' => self::COLUMN_STATUS, 'cond' => self::SQL_CONDITION_EQUAL, 'value' => iaCore::STATUS_ACTIVE]
        ];

        return $this->search($conditions, [self::COLUMN_ID, self::COLUMN_URL, self::COLUMN_TITLE, self::COLUMN_CODE],
            ['field' => self::NS_LEVEL, 'direction' => 'ASC']);
    }

    public function getChildren($parentId = self::ID_ROOT, $filterEmptyLocations = false)
    {
        $conditions = [
            ['column' => self::COLUMN_STATUS, 'cond' => self::SQL_CONDITION_EQUAL, 'value' => iaCore::STATUS_ACTIVE],
            ['column' => self::NS_PARENT, 'cond' => self::SQL_CONDITION_EQUAL, 'value' => $parentId]
        ];

        if ($filterEmptyLocations) {
            $conditions[] = ['column' => 'num_listings', 'cond' => '>', 'value' => 0];
        }

        return $this->search($conditions,
            [self::COLUMN_ID, 'num_listings' => 'num', self::COLUMN_URL, self::COLUMN_TITLE]);
    }

    public function getAllChildrenIds($id = self::ID_ROOT)
    {
        $parentNode = $this->_getNode($id);

        $conditions = '`' . self::NS_LEFT . '` >= ' . $parentNode[self::NS_LEFT] . ' AND `' . self::NS_RIGHT . '` <= ' . $parentNode[self::NS_RIGHT];

        return $this->iaDb->onefield(self::NS_NODE_ID, $conditions, 0, null, self::getTableTree());
    }

    public function getByUrl($url)
    {
        return $this->_get([
            self::COLUMN_STATUS => iaCore::STATUS_ACTIVE,
            self::COLUMN_URL => implode(IA_URL_DELIMITER, $url) . IA_URL_DELIMITER
        ],
        [iaDb::ALL_COLUMNS_SELECTION, self::NS_LEFT, self::NS_RIGHT, self::NS_LEVEL, self::NS_PARENT]);
    }

    // Manipulations with NS tree
    public function addRoot($title)
    {
        $nodeId = $this->iaDb->insert([self::COLUMN_ID => self::ID_ROOT, self::COLUMN_TITLE => $title], null,
            self::getTable());

        if ($nodeId) {
            $nodeData = [
                self::NS_NODE_ID => $nodeId,
                self::NS_PARENT => 0,
                self::NS_LEFT => 1,
                self::NS_RIGHT => 2,
                self::NS_LEVEL => 0
            ];

            $this->iaDb->insert($nodeData, null, self::getTableTree());
        }
    }

    public function insert(array $entryData)
    {
        $parentId = $entryData[self::NS_PARENT];
        unset($entryData[self::NS_PARENT]);

        // writing to the data table
        $nodeId = $this->iaDb->insert($entryData, null, self::getTable());

        if (empty($nodeId)) {
            return false;
        }

        $parentNode = $this->_getNode($parentId);

        $treeNode = [
            self::NS_NODE_ID => (int)$nodeId,
            self::NS_LEVEL => (int)($parentNode[self::NS_LEVEL] + 1),
            self::NS_LEFT => (int)($parentNode[self::NS_LEFT] + 1),
            self::NS_PARENT => (int)$parentId
        ];

        if ($entry = $this->iaDb->row('`' . self::NS_RIGHT . '`',
            '`parent` = ' . $parentId . ' ORDER BY `' . self::NS_LEFT . '` DESC', self::getTableTree())
        ) {
            $treeNode[self::NS_LEFT] = (int)$entry[self::NS_RIGHT] + 1;
        }

        $treeNode[self::NS_RIGHT] = $treeNode[self::NS_LEFT] + 1;

        // 'left' field then
        $values = [self::NS_LEFT => '`' . self::NS_LEFT . '` + 2'];
        $this->iaDb->update([], "`left` >= " . $treeNode[self::NS_LEFT], $values, self::getTableTree());

        // ... and 'right'
        $values = [self::NS_RIGHT => '`' . self::NS_RIGHT . '` + 2'];
        $this->iaDb->update([], "`right` >= " . $treeNode[self::NS_LEFT], $values, self::getTableTree());

        // inserting into nested set table
        $this->iaDb->insert($treeNode, null, self::getTableTree());

        return $this->iaDb->getAffected() ? $nodeId : false;
    }

    public function update(array $entryData, $id)
    {
        $parentId = $entryData[self::NS_PARENT];
        $nodeData = $this->_getNode($id);

        $result = true;

        if (!is_null($parentId) && $nodeData[self::NS_PARENT] != $parentId) {
            $parentData = $this->_get([self::COLUMN_ID => $parentId],
                [self::COLUMN_URL, self::NS_LEVEL, self::NS_LEFT]);

            $nodeEntry = [
                self::NS_LEFT => $parentData[self::NS_LEFT] + 1,
                self::NS_LEVEL => $parentData[self::NS_LEVEL] + 1,
                self::NS_PARENT => $parentId
            ];

            // NS tree manipulations
            $stmt = iaDb::printf('`left` >= :left AND `right` <= :right',
                ['left' => $nodeData[self::NS_LEFT], 'right' => $nodeData[self::NS_RIGHT]]);
            $nodeTree = (string)$this->iaDb->one('GROUP_CONCAT(`' . self::NS_NODE_ID . '`)', $stmt,
                self::getTableTree());

            if ($entry = $this->iaDb->row_bind([self::NS_RIGHT], '`parent` = :parent ORDER BY `left` DESC',
                ['parent' => $parentId], self::getTableTree())
            ) {
                $nodeEntry[self::NS_LEFT] = $entry[self::NS_RIGHT] + 1;
            }

            $childOffset = $nodeData[self::NS_RIGHT] - $nodeData[self::NS_LEFT] + 1;

            if ($nodeData[self::NS_LEFT] > $nodeEntry[self::NS_LEFT]) {    // moving to left
                $values = [self::NS_LEFT => '`' . self::NS_LEFT . '` + (' . $childOffset . ')'];
                $this->iaDb->update(null,
                    '`left` >= ' . $nodeEntry[self::NS_LEFT] . ' AND `left` <= ' . $nodeData[self::NS_LEFT] . ' AND `nid` NOT IN (' . $nodeTree . ')',
                    $values, self::getTableTree());

                $values = [self::NS_RIGHT => '`' . self::NS_RIGHT . '` + (' . $childOffset . ')'];
                $this->iaDb->update(null,
                    '`right` >= ' . $nodeEntry[self::NS_LEFT] . ' AND `right` <= ' . $nodeData[self::NS_RIGHT] . ' AND `nid` NOT IN (' . $nodeTree . ')',
                    $values, self::getTableTree());
            } else {
                $values = [self::NS_LEFT => '`' . self::NS_LEFT . '` - (' . $childOffset . ')'];
                $this->iaDb->update(null,
                    '`left` <= ' . $nodeEntry[self::NS_LEFT] . ' AND `left` >= ' . $nodeData[self::NS_LEFT] . ' AND `nid` NOT IN (' . $nodeTree . ')',
                    $values, self::getTableTree());

                $values = [self::NS_RIGHT => '`' . self::NS_RIGHT . '` - (' . $childOffset . ')'];
                $this->iaDb->update(null,
                    '`right` < ' . $nodeEntry[self::NS_LEFT] . ' AND `right` >= ' . $nodeData[self::NS_RIGHT] . ' AND `nid` NOT IN (' . $nodeTree . ')',
                    $values, self::getTableTree());
            }

            $level = $parentData[self::NS_LEVEL] - $nodeData[self::NS_LEVEL] + 1;
            $offset = $nodeData[self::NS_LEFT] - $nodeEntry[self::NS_LEFT];

            if ($nodeEntry[self::NS_LEFT] > $nodeData[self::NS_LEFT]) {
                $offset += $childOffset;
            }

            $values = [
                self::NS_LEFT => '`' . self::NS_LEFT . '` - (' . $offset . ')',
                self::NS_RIGHT => '`' . self::NS_RIGHT . '` - (' . $offset . ')',
                self::NS_LEVEL => '`' . self::NS_LEVEL . '` + (' . $level . ')'
            ];
            $this->iaDb->update(null,
                '`left` >= ' . $nodeData[self::NS_LEFT] . ' AND `right` <= ' . $nodeData[self::NS_RIGHT] . ' AND `nid` IN (' . $nodeTree . ')',
                $values, self::getTableTree());
            //

            $result = (bool)$this->iaDb->update($nodeEntry,
                iaDb::convertIds($nodeData[self::NS_NODE_ID], self::NS_NODE_ID), null, self::getTableTree());

            if ($result) {
                $entryData[self::COLUMN_URL] = $parentData[self::COLUMN_URL] . $entryData['alias'] . IA_URL_DELIMITER;
            }
        }

        if ($result) {
            unset($entryData[self::NS_PARENT]);
            $this->iaDb->update($entryData, iaDb::convertIds($id), null, self::getTable());

            $result = (0 === $this->iaDb->getErrorNumber());
        }

        return (bool)$result;
    }

    public function delete($categoryId) // removes children as well
    {
        $nodeData = $this->_getNode($categoryId);

        $stmt = iaDb::printf('`:column` >= :left AND `:column` <= :right',
            ['column' => self::NS_LEFT, 'left' => $nodeData[self::NS_LEFT], 'right' => $nodeData[self::NS_RIGHT]]);
        $entriesList = $this->iaDb->onefield(self::NS_NODE_ID, $stmt, null, null, self::getTableTree());

        $pattern = '`:column` IN (' . implode($entriesList, self::VALUES_DELIMITER_CHAR) . ')';

        $stmt = iaDb::printf($pattern, ['column' => self::COLUMN_ID]);
        $dataEntries = $this->iaDb->all(iaDb::ALL_COLUMNS_SELECTION, $stmt, null, null, self::getTable());

        if (!$this->iaDb->delete($stmt, self::getTable())) {
            return false;
        }

        if (isset($dataEntries[0]['pictures'])) {
            $iaPicture = $this->iaCore->factory('picture');
            foreach ($dataEntries as $entry) {
                if ($entry['pictures']) {
                    $iaPicture->delete($entry['pictures']);
                }
            }
        }

        $stmt = iaDb::printf($pattern, ['column' => self::NS_NODE_ID]);
        $this->iaDb->delete($stmt, self::getTableTree());

        $offset = ($nodeData[self::NS_RIGHT] - $nodeData[self::NS_LEFT]) + 1;

        // updating 'left' field
        $rawValues = [
            self::NS_LEFT => '`' . self::NS_LEFT . '` - ' . $offset
        ];
        $stmt = iaDb::printf('`:column` > :value', ['column' => self::NS_LEFT, 'value' => $nodeData[self::NS_RIGHT]]);

        $this->iaDb->update(null, $stmt, $rawValues, self::getTableTree());

        // updating 'right' field
        $rawValues = [
            self::NS_RIGHT => '`' . self::NS_RIGHT . '` - ' . $offset
        ];
        $stmt = iaDb::printf('`:column` > :value', ['column' => self::NS_RIGHT, 'value' => $nodeData[self::NS_RIGHT]]);

        $this->iaDb->update(null, $stmt, $rawValues, self::getTableTree());

        return true;
    }

    //

    protected function _editCounter($entryId, $action = self::COUNTER_INCREMENT)
    {
        $parentsList = $this->getParentsFlat($entryId);

        if (empty($parentsList)) {
            return false;
        }

        return $this->iaDb->update(null,
            '`id` IN (' . $parentsList . ')',
            ['num_listings' => '`num_listings` ' . $action . ' 1'],
            self::getTable()
        );
    }

    public function editCounters($action, array $itemData, $previousData = null)
    {
        switch ($action) {
            case iaCore::ACTION_EDIT:
                if (!isset($itemData['location_id'])) {
                    if (iaEstate::STATUS_AVAILABLE == $previousData['status'] && iaEstate::STATUS_AVAILABLE != $itemData['status']) {
                        $this->_editCounter($previousData['location_id'], self::COUNTER_DECREMENT);
                    } elseif (iaEstate::STATUS_AVAILABLE != $previousData['status'] && iaEstate::STATUS_AVAILABLE == $itemData['status']) {
                        $this->_editCounter($previousData['location_id']);
                    }
                } else {
                    if ($itemData['location_id'] == $previousData['location_id']) {
                        if (iaEstate::STATUS_AVAILABLE == $previousData['status'] && iaEstate::STATUS_AVAILABLE != $itemData['status']) {
                            $this->_editCounter($itemData['location_id'], self::COUNTER_DECREMENT);
                        } elseif (iaEstate::STATUS_AVAILABLE != $previousData['status'] && iaEstate::STATUS_AVAILABLE == $itemData['status']) {
                            $this->_editCounter($itemData['location_id']);
                        }
                    } else { // category changed
                        iaEstate::STATUS_AVAILABLE == $itemData['status']
                            && $this->_editCounter($itemData['location_id']);
                        iaEstate::STATUS_AVAILABLE == $previousData['status']
                            && $this->_editCounter($previousData['location_id'], self::COUNTER_DECREMENT);
                    }
                }
                break;
            case iaCore::ACTION_ADD:
                iaEstate::STATUS_AVAILABLE == $itemData['status']
                    && $this->_editCounter($itemData['location_id']);
                break;
            case iaCore::ACTION_DELETE:
                iaEstate::STATUS_AVAILABLE == $itemData['status']
                    && $this->_editCounter($itemData['location_id'], self::COUNTER_DECREMENT);
        }
    }

    public function rebuild()
    {
        $locations = $this->iaDb->all([iaDb::ID_COLUMN_SELECTION, 'alias', self::COLUMN_URL], '`id` > ' . self::ID_ROOT, null,
            null, self::getTable());

        $aliasMap = [];
        $urlMap = [];

        // first, populate the data mapping structures
        foreach ($locations as $entry) {
            $aliasMap[(int)$entry[iaDb::ID_COLUMN_SELECTION]] = $entry['alias'];
            $urlMap[(int)$entry[iaDb::ID_COLUMN_SELECTION]] = explode(IA_URL_DELIMITER,
                trim($entry[self::COLUMN_URL], IA_URL_DELIMITER));
        }
        $urlMap[self::ID_ROOT] = [];

        $parentsMap = [self::ID_ROOT => 0];
        foreach ($urlMap as $id => $urlChunks) {
            $parentId = 0;

            if ($urlChunks) {
                $parentId = self::ID_ROOT;
                array_pop($urlChunks); // remove the alias for further processing
                if ($urlChunks) {
                    $i = 0;
                    while ($i < count($urlChunks) && isset($urlChunks[$i])) // start to look for respective parent node
                    {
                        $parentId = array_search($urlChunks[$i], $aliasMap);
                        $i++;
                    }
                }
            }

            $parentsMap[$id] = $parentId;
        }

        $children = [];
        foreach ($parentsMap as $id => $parentId) {
            $children[$parentId][] = $id;
        }

        function processNodes($nodeId, $children, &$lefts, &$rights, &$counter)
        {
            $counter++;

            $lefts[$nodeId] = $counter;
            $subnodesCount = empty($children[$nodeId]) ? 0 : count($children[$nodeId]);

            if ($subnodesCount) {
                $i = 0;
                while (isset($children[$nodeId][$i])) {
                    $subnodesCount += processNodes($children[$nodeId][$i], $children, $lefts, $rights, $counter);
                    $i++;
                }
            } else {
                $counter++;
            }

            $rights[$nodeId] = $lefts[$nodeId] + (2 * $subnodesCount) + 1;

            if (isset($i)) {
                $counter++;
            }

            return $subnodesCount;
        }

        $counter = 0;
        $lefts = $rights = [];

        processNodes(self::ID_ROOT, $children, $lefts, $rights, $counter);

        $sql = '';
        foreach ($parentsMap as $id => $parentId) {
            // node id, parent, left, right, level
            $sql .= sprintf(
                '(%d,%d,%d,%d,%d),',
                $id, $parentId, $lefts[$id], $rights[$id], count($urlMap[$id])
            );
        }
        $sql = substr($sql, 0, -1); // remove the last comma

        $this->iaDb->truncate(self::getTableTree(true));

        // we use a manually prepared single query since the iaDb::insert()...
        // usually executes the INSERT instruction for each entry
        $sql = 'INSERT INTO `' . self::getTableTree(true) . '` VALUES' . $sql;
        $this->iaDb->query($sql);
    }

    /*
        public function rebuild()
        {
            $this->iaDb->truncate($this->iaDb->prefix . self::getTableTree());

            $locations = $this->iaDb->all(iaDb::ALL_COLUMNS_SELECTION, '1 ORDER BY `id`', 0, null, self::getTable());
            $locations_num = count($locations);
            foreach ($locations as $location)
            {
                $parents = explode(IA_URL_DELIMITER, $location['url']);
                unset($parents[count($parents) - 1]);
                $parentUrl = implode(IA_URL_DELIMITER, $parents) . IA_URL_DELIMITER;

                $fields['nid'] = $location['id'];
                $fields['level'] = $this->iaDb->one("LENGTH(`url`) - LENGTH(REPLACE(`url`, '/', '' ))", "`id` = " . $location['id'], self::getTable());

                if ($fields['level'] > 1)
                {
                    $fields['parent'] = $this->iaDb->one('id', "`url` = '" . $parentUrl . "'", self::getTable());
                }
                elseif ($fields['level'] == 0)
                {
                    $fields['parent'] = 0;
                }
                else
                {
                    $fields['parent'] = 1;
                }

                $this->iaDb->insert($fields, null, self::getTableTree());
            }

            //set left and right
            $this->iaDb->update(array(self::NS_LEFT => 1), '`level` = 0', array(self::NS_RIGHT => $locations_num * 2), self::getTableTree());

            $last_lvl = (int)$this->iaDb->one('MAX(`level`)', null, self::getTableTree());

            $levels = array();

            for ($lvl = 1; $lvl <= $last_lvl; $lvl++)
            {
                $where = "`level` = '" . $lvl . "' ORDER BY `nid`";
                if ($lvl == 1)
                {
                    $locs1 = $this->iaDb->all(array(self::NS_NODE_ID, self::NS_PARENT), $where, 0, null, self::getTableTree());
                    $left = 2;
                    $right = 3;
                    foreach ($locs1 as $loc)
                    {
                        $levels[$lvl][$loc[self::NS_NODE_ID]] = array(
                            self::NS_LEFT => $left,
                            self::NS_RIGHT => $right,
                            self::NS_PARENT => $loc[self::NS_PARENT]
                        );

                        $left += 2;
                        $right += 2;
                    }
                }
                else
                {
                    $where = "`level` = '" . $lvl . "' ORDER BY `parent`";

                    $locs = $this->iaDb->all(array(self::NS_NODE_ID, self::NS_PARENT), $where, null, null, self::getTableTree());

                    foreach ($locs as $loc)
                    {
                        if (isset($levels[$lvl - 1][$loc[self::NS_PARENT]]['current_right']))
                        {
                            $levels[$lvl][$loc[self::NS_NODE_ID]] = array(
                                self::NS_LEFT => $levels[$lvl - 1][$loc[self::NS_PARENT]]['current_right'] + 1,
                                self::NS_RIGHT => $levels[$lvl - 1][$loc[self::NS_PARENT]]['current_right'] + 2,
                                self::NS_PARENT => $loc[self::NS_PARENT]
                            );
                        }
                        else
                        {
                            $levels[$lvl][$loc[self::NS_NODE_ID]] = array(
                                self::NS_LEFT => $levels[$lvl - 1][$loc[self::NS_PARENT]][self::NS_LEFT] + 1,
                                self::NS_RIGHT => $levels[$lvl - 1][$loc[self::NS_PARENT]][self::NS_LEFT] + 2,
                                self::NS_PARENT => $loc[self::NS_PARENT]
                            );
                        }

                        $levels[$lvl - 1][$loc[self::NS_PARENT]]['current_right'] = $levels[$lvl][$loc[self::NS_NODE_ID]][self::NS_RIGHT];

                        $parentId = $loc[self::NS_PARENT];

                        for ($i = $lvl - 1; $i > 0; $i--)
                        {
                            $levels[$i][$parentId][self::NS_RIGHT] += 2;

                            foreach ($levels[$i] as $pid => $lvl_parent)
                            {
                                if ($lvl_parent[self::NS_LEFT] > $levels[$i][$parentId][self::NS_LEFT])
                                {
                                    $levels[$i][$pid][self::NS_LEFT] += 2;
                                    $levels[$i][$pid][self::NS_RIGHT] += 2;
                                }
                            }

                            $parentId = $levels[$i][$parentId][self::NS_PARENT];
                        }
                    }
                }
            }

            foreach ($levels as $level)
            {
                foreach ($level as $nid => $node)
                {
                    unset($node['current_right']);
                    $this->iaDb->update($node, iaDb::convertIds($nid, 'nid'), null, self::getTableTree());
                }
            }
        }
    */
    public function getSitemapEntries()
    {
        $result = [];

        if ($locations = $this->search([
            [
                'column' => self::COLUMN_STATUS,
                'cond' => self::SQL_CONDITION_EQUAL,
                'value' => iaCore::STATUS_ACTIVE
            ]
        ], ['url'])
        ) {
            foreach ($locations as $entry) {
                $result[] = $this->getUrl($entry);
            }
        }

        return $result;
    }
}
